;*********************************************************************
; File:     RoboCar_US_Measure.ASM
; Date:     17.08.32023
; Author:   Ottmar
; Copyright unbeschrnkt freigegeben
; Zweck:    Darstellung und Aufbereitung einer Entfernungsmessung
;           mit Ultraschallmodul HC-SR04
;           16Bitx16Bit Multiplikation
;           32Bit Binr zu Dezimalwandung
;           Das simulierte Messergebnis wird im Debuggmodus im
;           MPLAB-Watch-Fenster (View-Watch) mit dem Variablen
;           BCD9..BCD0 dargestellt.
;           3452 working Cycles (wc) Ausfhrungszeit: 1,726ms
;*********************************************************************
;  Prozessor:  PIC 16F1936 (mu im Configure-Men gesetzt werden)
;  Hardware:   nicht erforderlich-Simulation im Codefenster
;              fosc = 8MHz
   ;
   ERRORLEVEL -205 ;Found directive in column 1.
   ERRORLEVEL -207 ;Found label after column 1.
   ERRORLEVEL -302 ;Register in operand not in bank 0. Ensure ..
   ERRORLEVEL -303 ;Program word too large. Truncated to core size
;                  ;DON'T CARE about this message!!
;--MCU DEFINITION
   LIST     P=16F1936      ;list directive to define processor
   #INCLUDE <P16F1936.INC> ;processor specific variable definitions
;--MCU CONFIGURATION
   __CONFIG _CONFIG1, _FOSC_INTOSC &  _WDTE_OFF & _CP_OFF & _CPD_OFF & _MCLRE_ON & _PWRTE_OFF & _BOREN_OFF & _IESO_OFF & _FCMEN_OFF;
   __CONFIG _CONFIG2, _PLLEN_OFF  & _LVP_OFF &_WRT_OFF & _VCAPEN_OFF & _STVREN_OFF & _BORV_19;
;   System-clock is INTOSC
;   FOSC_INTOSC I int. osc.enabled I/O function on CLKIN pin
;   PLLEN_OFF     4x PLL enabled/disabled
;*********************************************************************
CBLOCK 0x20
   A,bit,u2,u1          ;nmac Macro zu. MULT_16x16            
   ACb1, ACb0           ;MULT_16x16 1. Multiplikant
   BCb1, BCb0           ;           2. Multiplikator            
   CCb3,CCb2            ;Produkt    32Bit CCb3:CCb0
   CCb1,CCb0
   tmp
   ;
   BCD9,BCD8,BCD7,BCD6  ;Bin16_Dez Ergebnisvariable
   BCD5,BCD4,BCD3,BCD2  ;          BDC9:0 
   BCD1,BCD0
   ACb3,ACb2            ;Bin32_Dez bergabevariable
 ;  ACb1,ACb0           ;          vgl. oben MULT16x6  
   cntBit,cntDig
   ;
   flgObstacle          ;Flags zr Hinderniserkennung
   d4,d3,d2,d1          ;countwer in delays
ENDC
;
copy MACRO  Source, Target ; e.g. PIC18 ASM "MOVFF A,B"
   movf     Source,w       ;how to use: copy Var1,Var2
   movwf    Target
   ENDM   
   ;
;*********************************************************************
;--LABELS & CONSTANTS
;*********************************************************************
;--US_PORT (PORTC)            ;Distance measurement using US module
   US_PORT     EQU   PORTC     
   US_LAT      EQU   LATC
   US_TRIS     EQU   TRISC
   TRIG        EQU   RC5      ;(16) triggers US-Burst
   ECHO        EQU   RC4      ;(15) receive US-Echo
   IRL         EQU   RC3      ;(14) IR-Sensor Links
   ;                 RC2      ;(13) PWM2 output
   ;                 RC1      ;(12) PWM1 output
   IRR         EQU   RC0      ;(11) IR-Sensor rechts
   #DEFINE  US_TRIG  US_LAT,TRIG
   ;
;--flgObstacle used to avoid obstacles 
   OBST_US     EQU   .7       ;=1obstacle in FRONT
   JUMPOVR     EQU   .1        ;berspringt die reale US-Messung
   OVFLW       EQU   .0       ;Overflow TMR1 (US-Distance)
;*********************************************************************
   ORG      0x0000      ;processor reset vector
   GOTO     init      
;*********************************************************************
; INTERRUPT SERVICE ROUTINE (currently not used)
   ORG      0x0004      ;Interrupt Vector
   RETFIE
;*********************************************************************
init:
;  Calculation of the multiplier, adapting delays to fosc
   FOSC        EQU    d'8'       ;fosc extern/intern
   FOSC_MULT   EQU    FOSC/d'4'  ;instrucktions -> fosc/4
   ;
;--INTOSC                                                
	BANKSEL	OSCCON			                        ;S.75 (PLL) 
	clrf	   OSCCON                                 ;S.82
	bcf		OSCCON,SPLLEN	   ;b7=0/1 =1 4xPLL is disabled/enabled
	bsf		OSCCON,IRCF3	   ;b6:3 =1 1110 IntOsc=8MHz[*4xPLL = 32MHz]
	bsf		OSCCON,IRCF2	   ;b5   =1
	bsf		OSCCON,IRCF1	   ;b4   =1
	bcf		OSCCON,IRCF0	   ;b3   =0   
							         ;b2   =0 unimplemented
	bsf		OSCCON,SCS1		   ;b1	=1 System Clock is int.osc.
	bcf		OSCCON,SCS0		   ;b0   =0 ignored if b1=1
	;
;INITIALIZATION OF US-MODULE
;Set Trig and Echo Low to initalize module. Place a minimum 10us high
;level pulse to "Trigger" (module will automatically send eight 40KHz
;acoustic bursts). At the same time, gate the microcontroller timer
;to start timing.
;See also subroutine "US_Measure"
   BANKSEL  US_PORT        ;PORTB
   clrf     US_PORT
   BANKSEL  US_LAT
   clrf     US_LAT
;   bcf      US_LAT,ECHO    ;set ECHO = LOW
;   bcf      US_LAT,TRIG    ;set TRIG = LOW
   BANKSEL  US_TRIS        ;bank2 write to LAT read from PORT!
   clrf     US_TRIS
;   bcf      US_LAT,ECHO    ;set ECHO, TRIG = as output 
;   bcf      US_LAT,TRIG    ;initialize ECHO as digital Input 
   BANKSEL  0
   ;
   clrf     flgObstacle
   bcf      T1CON,TMR1ON    ;TMR1 is now disabled
   clrf     TMR1H
   clrf     TMR1L
   bcf      PIR1,TMR1IF    ;b0=0 TMR1 has not overflowed
   ;
   GOTO main
;*********************************************************************
;  MAIN PROGRAMM
;*********************************************************************
main:
;Das Programm simuliert die US-Abstandsmessung, berspringt aber die 
;eigentliche Messung (Auslung des US-Burst, Empfang des Echos)indem
;kein US-Modul angeschlossen ist.
   ;
   bsf      flgObstacle,JUMPOVR  ;b1=1 berspringt die reale US-Messung
   CALL     US_Measure           ;measuere, calluculate -> distance
   CALL     US_Distance          ; =752[,]89886mm
   nop
   ;
main_end:
   GOTO  main_end
   GOTO  main
;*********************************************************************
;  SUBROUTINES
;*********************************************************************
Test_EchoTime:    ;This subis called in the "US_Measure" subprogram
; fosc = 8MHz
; Die Messzeit bezieht sich auf den gesamenten Schallweg, daher der
; Multiplikator "*2"
; Distanz zum Hindernis: 753mm  Echoweg = 753mm *2 
; TMR1H:L = (753mm *2) /(0,171645mm/0,5us) = 8773 = 0x2245
   movlw    0x22
   movwf    TMR1H
   movlw    0x45
   movwf    TMR1L
;; Abstand 287mm *2 -> TMr1H:L = 574+0000 /858 = 6689 = 0x1A21
;   movlw    0x0D
;   movwf    TMR1H
;   movlw    0x11
;   movwf    TMR1L
   RETURN
;*********************************************************************   
US_Measure:
; Measurement of the delay time between US burst and US echo.
; 1. Trigger an US-BURST         set Trigger-Pin to LOW
; 2. START TMR1                  fos/4 = 2MHz -> 1,715mm/working cycle
; 3. Waiting for the US-Echo     until Echo-Pin is set to HIGH
; 4. STOP TMR1                   every usec, TMR1 was increased by 2
; 5. Calculate distance
; 6. If an obstacle is closer then 30cm, a flag is set.
; 
; v_schall(m/us) = 343,29(m/s) / 1.000.000  = 0,000343m/us 
;   
; GIVEN: v_schall=343,29m/s, fosc = 8MHz -> fosc/4 = 2MHz
; TMR1H:L increments each 0.5us 
; s_schall mm/0,5us: 343,29m*1000 / 2.000.000Hz =  0,171645mm/0,5us
; TMR1H:L bei s_schall=1000mm = 1000mm /0,171645mm/0,5u = 5825
; for half the sound path one can calculate:
; 1/1 s_schall = (TMR1H:L=5825) * 0,17164*0,5 = 998mm
; 1/2 s_schall =          5825  * 0,0858225   = 499,9 mm                                             =
; simplyfied:             5825  * 8582        = 500,[07625]mm
; When converting binary to decimal, a 16Bitx16Bit multiplication
; results in a product with a maximum of 10 digits. In order to 
; output the distance to the obstacle in mm, the last 4 digits
; e.g. [07625] of the decimal number are simply not displayed.
;*********************************************************************
; TMR1 starts at 0x0000, then measuring propagation delay of US-signal
; No prescaler is used, the duty cycle fosc/4 is incremented directly
; by TMR1. Measurement is discarded if TMR1 overflows
   ;
   bcf      T1CON,TMR1ON      ;make sure that TMR1 is disabled   
   clrf     TMR1H             ;set TMR1 to 0x0000
   clrf     TMR1L
   bcf      PIR1,TMR1IF       ;b0=0 TMR1 has not overflowed
   bcf      flgObstacle,OVFLW ;clear TMR1 has not overflowed
   ;
;  Set TRIGGER =1 for min. 11wc
   BANKSEL  US_LAT            ;RC5(16) trigger now an US-Burst
   bsf      US_LAT,TRIG       ;RC5=1!
   BANKSEL  0
   movlw    .5*FOSC_MULT      ;delay-counter delaytime=52wc=13us
   movwf    d4
us_trig_loop:
   decfsz   d4,f
   GOTO     us_trig_loop
   BANKSEL  US_LAT
   bcf      US_TRIG           ;RC5=0 HC_SR04 sends an US-BURST
   BANKSEL  0
   ;
;---TEST------------------------------------------------
; Testcode eingefgt um auch ohne US-Module Berechnungen durchzufhren
; JUMPS OVER THE US RUNTIME MEASUREMENT
   btfss    flgObstacle,JUMPOVR
   GOTO     usmeas_echo
   CALL     Test_EchoTime     ;Holt TMR1H:L Testdaten
   GOTO     usmeas_end        ;JUMP OVER RUNTIME MEASUREMENT
;---TESTENDE---------------------------------------------
usmeas_echo:                  ;Wait for rising edge on ECHO-pin RC4(15) 
   btfss    US_PORT,ECHO 
   GOTO     usmeas_echo       ;wait until ECHO-pinis LOW
   bsf      T1CON,TMR1ON      ;start TIMER 1
   ;
   movlw    .8*FOSC_MULT      ;
   CALL     Delay1ms          ;TMR1 0xD84A = 63562

usmeas_loop:
   btfsc    US_PORT,ECHO      ;RA6=1?  Echo eingetroffen?
   GOTO     usmeas_loop       ;NO,  try it again
   bcf      T1CON,TMR1ON      ;YES, b7=0 TMR1 STOP
usmeas_end:   
   btfsc    PIR1,TMR1IF       ;b0=0 TMR1 has not overflowed
   bsf      flgObstacle,OVFLW ;TMR1 has overflowed no obstacle in front  
   ;                          ;US-Modul nicht verbunden                
   RETURN
;*********************************************************************
US_Distance:
; Calculates the distance to an obstacle.
;---------------------------------------------------------------------
; f   frequency(Hz),  us  microseconds
; wc  "working cycle" 1 Arbeitstakt des PIC16F1936 = 1s/(fosc/4)
;
; Given: v_schall 20C = 343,29m/s
;        t_echo 2m  = 2m/343,29 = 5,8261-3s
; Calc: 
;        TMR1H:L    = 2*10-6Hz  * 5,8261-3s = 11652wc
;        s_echo mm  = 11652 * 0,1716mm/wc   = 1999,8mm
;        dist. to Obstacle =  s_echo/2      = 1998mm/2 = 999mm
;
; Simplification:: 
;        TMR1H:L * 0,5Distance/wc = 0,1715/2==0,0857,5
;        TMR1H:L * 857 = distance to obstacle
;        11652   * 0,0858mm/wc   = 999,7416.mm
;        11652   * 858           = 9997416
;
; Solution: 
; We simply cut off the last 4 digits (e.g. no output on the LCD) and
; thus get the distance to the obstacle in mm!
; --------------------------------------------------------------------
; Check TMR1- has the minimum distance been observed?
; 300mm +4Digits-> 300+0000 /858 = 3496 = 0x0DA8
   ;
   copy     TMR1H,ACb1   ;Multiplikant in MULT_16x16
   movwf    ACb1
   copy     TMR1L, ACb0
; Minimum distance below 300mm? -> then set flag
   bcf      flgObstacle,OBST_US
   movlw    0x0D        ;HByte
   subwf    TMR1H,w     ;      Actual value minus minimum distance
   btfss    STATUS,C    ;      C=0? distance < minimum distance?
   bsf      flgObstacle,OBST_US;JA, Y_ERR=1 Obstacle detected ahead
; If the H byte is smaller than the minimum distance, the low byte 
; no longer needs to be checked.
; If the H byte is the same size as that of the minimum distance, the
; Lo byte is also checked.    
   btfss    STATUS,Z    ;Z=1? Had both H bytes the same value?
   GOTO     usdist_mult ;      No
   movlw    0xA8;       ;
   subwf    TMR1L,w     ;
   btfss    STATUS,C    ;C=0? Low byte actual value is smaller?
   bsf      flgObstacle,OBST_US; set Flag
   ;
; Subsequent multiplication can be omitted if no LCD is used to
; display the distance to the obstacle.
   ;
usdist_mult:
; A multiplication of 16 bits x 10 bits would also be sufficient
; to calculate the distance.
; BCb1:0 is the fixed Multiplikator to calculate result in mm
   ;
   movlw    0x21           ;FOSC=8MHz 8582 BCb1:0 = 0x2186
   movwf    BCb1           ;FOSC=4MHz 4291 BCb1:0 = 0x10C3
   movlw    0x86
   movwf    BCb0           ;0x10C3 =4291 Multiplikator in MULT_16x16
   CALL     MULT16x16      ;Ergebnis 32Bit CCb3:CCb0 (36us=144cyc)
   ;                       0x00.98.B2.9B=10007195
   copy     CCb3, ACb3     ;copy product to transfer variables oft
   copy     CCb2, ACb2     ;sub Bin32_Dez
   copy     CCb1, ACb1
   copy     CCb0, ACb0
   CALL     Bin32_Dez      ;result in BCD9:0= 1000,719mm

   movlw    .9 -.1         ;-1, BCD0 wird immer angezeigt 
   movwf    cntDig
   movlw    BCD9           ;Pointer auf hchste Dezimalstelle
;;;   CALL     LeadZero_Lcd   ;fhrende Nullen entfernen
   RETURN                  ;Ausgabe im LCD nur BCD9:6 mm 
;*********************************************************************

;*********************************************************************
MULT16x16:
; Code found on page: 
; http://www.piclist.com/techref/microchip/math/radix/b2bp-32b10d.htm
; 16bit by 16bit unsigned multiply result 32Bit
; by Martin Sturm 2010 tested 140 instructions, 140 cycles
;
;CBLOCK
;   A,bit,u2,u1         ;nmac Macro zu. MULT_16x16            
;   ACb1, ACb0          ;MULT_16x16 1. Multiplikant
;   BCb1, BCb0          ;           2. Multiplikator            
;   CCb3,CCb2           ;Produkt    32Bit CCb3:CCb0
;   CCb1,CCb0
;ENDC
   ;  
mmac  MACRO    A,bit,u2,u1   ;Hilfs-Macro zu MULT_16x16
      btfsc    A,bit
      addwf    u2,f
      rrf      u2,f
      rrf      u1,f
   ENDM
;
;MULT_16x16_FASTEST MACRO ACb1,ACb0, BCb1,BCb0, CCb3,CCb2,CCb1,CCb0, tmp
   ; 16x8 multiply  ACb1:ACb0 * BCb0 -> CCb2:CCb3:CCb0
   clrf  CCb2
   clrf  CCb0
   bcf   STATUS,C                   ;Clear Carry 
   movf  BCb0,w
   mmac  ACb0,0, CCb2,CCb0
   mmac  ACb0,1, CCb2,CCb0
   mmac  ACb0,2, CCb2,CCb0
   mmac  ACb0,3, CCb2,CCb0
   mmac  ACb0,4, CCb2,CCb0
   mmac  ACb0,5, CCb2,CCb0
   mmac  ACb0,6, CCb2,CCb0
   mmac  ACb0,7, CCb2,CCb0
   CLRF  CCb3
   ; carry already clear from last RRF of mmac above
   ; BCb0 still in W
   mmac  ACb1,0, CCb2,CCb3
   mmac  ACb1,1, CCb2,CCb3
   mmac  ACb1,2, CCb2,CCb3
   mmac  ACb1,3, CCb2,CCb3
   mmac  ACb1,4, CCb2,CCb3
   mmac  ACb1,5, CCb2,CCb3
   mmac  ACb1,6, CCb2,CCb3
   mmac  ACb1,7, CCb2,CCb3
   ; 16x8 multiply  ACb1:ACb0 * BCb1 -> CCb3:tmp:CCb1
   ; middle byte from previous 16x8 multiplication starts in CCb3
   ; but ends in CCb1
   clrf  CCb1
   movf  BCb1,w
   ; carry already clear from last RRF of mmac above
   mmac  ACb0,0, CCb3,CCb1
   mmac  ACb0,1, CCb3,CCb1
   mmac  ACb0,2, CCb3,CCb1
   mmac  ACb0,3, CCb3,CCb1
   mmac  ACb0,4, CCb3,CCb1
   mmac  ACb0,5, CCb3,CCb1
   mmac  ACb0,6, CCb3,CCb1
   mmac  ACb0,7, CCb3,CCb1
   ; W still holds BCb1
   clrf  tmp   ; BCb1,BCb0, or ACb0 may be used for tmp
   ; carry already clear from last RRF of mmac above
   mmac  ACb1,0, CCb3,tmp
   mmac  ACb1,1, CCb3,tmp
   mmac  ACb1,2, CCb3,tmp
   mmac  ACb1,3, CCb3,tmp
   mmac  ACb1,4, CCb3,tmp
   mmac  ACb1,5, CCb3,tmp
   mmac  ACb1,6, CCb3,tmp
   mmac  ACb1,7, CCb3,tmp
   ; add middle byte ACb1:ACb0 * BCb1 to upper byte of ACb1:ACb0 * BCb0
   movf  tmp,w
   addwf CCb2,f
   btfsc STATUS,C
   incf  CCb3,f
   RETURN
;*********************************************************************
; Bin32_Dez   -   32-bit unsigned BINARY to BCD conversion
; HINWEISE ZUR VERWENDUNG B32-24-B16-B8 ZU DEZIMAL
;*********************************************************************
; bergabevariable:  ACb3:0
; Ergebnisvariable:  BCD9:0 (Steht im FSR)
; Achtung! Midrange PIC anstatt FSR,IND -> FSR0, INDF0 verwenden
;          PIC18F45K22 anstatt  FSR1:0  -> FSR2H:L INDF2:0 verwenden
;                                          FSR1H:L
;  
; Die Ergebnisvariablen BCD9:00 werden per FSR zugewiesen. Daher
; mu das hchste Digit, z.B. BCD9 an der niedrigsten Adresse stehen
; Z.B. 0x37, BCD9...0 jeweil an der um 1 incrementierten Adresse, bis
; BCD0 an 0x37+0x9A
; BCD9 an der niedrigsten Adresse z.B. 0x37      = d'55'
; BCD0 an der hchsten Adresse    z.B. 0x40+0x09 = d'64'
;
; Anpassung an 24/16/8Bit
; Es knnen wahlweise 32-24-16-Bit zu Dezimal gewandelt 
; werden. Dazu sind diese nderungen vorzunehmen:
; Bits   cntBit   cntDig   bergabe:
; 32Bit  .32      .10      ACb3:0
; 16Bit  .16      .5       ACb1:0   auskommentieren: ACb3:2
;  8Bit   .8      .3       ACb0     "                ACb3:1
; nicht verwendete Variablen BCD9)...BCD1 evtl auskommentieren.
;********************************************************************
Bin32_Dez:              ;vgl obige Anmerkungen
; Code found on page: 
; http://www.piclist.com/techref/microchip/math/radix/b2bp-32b10d.htm
; with respect to: KILLspamwalter.piel at -Remove-sasol.com
;
; Achtung! Midrange PIC anstatt FSR,IND -> FSR0, INDF0 verwenden
;          PIC18F45K22 anstatt FSR1:0   -> FSR2H:L INDF2:0 verwenden
;                                          FSR1H:L
;  CBLOCK                                  FSR0H:L
;      BCD9,BCD8,BCD7,BCD6    ;BDC9:0 Ergebnisvariable
;      BCD5,BCD4,BCD3,BCD2
;      BCD1,BCD0
;      ACb3,ACb2,ACb1,ACb0    ;ACb3:0 bergabevariable
;      cntBit,cntDig
;  ENDC   

   clrf     BCD9        ;10^9 1 Milliarde MSB 32bit zu Dezimal
   clrf     BCD8        ;10^8
   clrf     BCD7        ;10^7
   clrf     BCD6        ;10^6 Million
   clrf     BCD5        ;10^5
   clrf     BCD4        ;10^4
   clrf     BCD3        ;10^3
   clrf     BCD2        ;Hunderter
   clrf     BCD1        ;Zehner
   clrf     BCD0        ;Einer
   ;---------Outer loop
   movlw    d'32'       ;BitCounter 32Bit (16Bit->movlw d'16' ..... 
   movwf    cntBit      ;bit counter
b2bcd1:
   rlf      ACb0,f      ;Shift 32-bit accumulator
   rlf      ACb1,f      ;left to put MSB into Carry
   rlf      ACb2,f
   rlf      ACb3,f
   movlw    BCD0        ;Point to address of least 
   movwf    FSR0         ;significant bcd digit
   ;
   movlw    d'10'       ; 32Bit 
   movwf    cntDig      ; digit counter
      ;---------Inner loop
b2bcd2:
   rlf      INDF0,f     ;Shift Carry into bcd digit
   movlw    D'10'       ;Subtract ten from digit then
   subwf    INDF0,w     ; check and adjust for decimal overflow
   btfsc    STATUS,C    ;If Carry=1 (result >= 0)
   movwf    INDF0       ; adjust for decimal overflow
   decf     FSR0,f      ;Point to next bcd digit
   decfsz   cntDig,f    ;Decrement digit counter
   GOTO     b2bcd2      ; - go if cntDig > 0
   decfsz   cntBit,f    ;Decrement bit counter
   GOTO     b2bcd1      ; - go if cntBit > 0
   RETLW    0
;*********************************************************************
Delay1ms:
; 1000 working cycles incl. CALL
; Aufruf: movlw   Literal*FOSC_MULT ;Literal = (Zahl*FOSC_MULT) <=255
;         CALL    Delay1ms        ;FOSC_MULT vgl. ';LABELS and CONSTANTS'
   ;
   movwf    d3          ;Multiplikator fr Basisdelay
   xorlw    .0          ;Vergleich WREG = d3 = 0 
   btfsc    STATUS,Z    ;Z=1? WREG=0? 
   incf     d3,f        ;JA, d3+1 -> 10ms Delayzeit
   ;
delay1ms2:
   movlw    .13
   movwf    d2
delay1ms1:          
   movlw    .10
   movwf    d1
delay1ms0:
   nop      ;6 x nop=10 cycles/durchlauf
   nop
   nop
   decf     d1,f
   btfss    STATUS,Z
   GOTO     delay1ms0
   nop
   decf     d2, f
   btfss    STATUS,Z
   GOTO     delay1ms1
   decf     d3,f
   btfss    STATUS,Z
   GOTO     delay1ms2
   RETURN
;*********************************************************************
   END